<?php

namespace App\Services;

use Plivo\RestClient;
use Plivo\Resources\Message\MessageCreateResponse;
use Plivo\Resources\Message\MessageCreateErrorResponse;
use Illuminate\Support\Facades\Cache;
use App\Http\Controllers\Fast2SmsController;

class OtpServices
{
    // Simple debug logger function
    private function logToFile($data, $filename = 'otp-service.txt')
    {
        $file = storage_path('app/' . $filename);
        $content = "[" . date('Y-m-d H:i:s') . "]\n" . (is_string($data) ? $data : json_encode($data, JSON_PRETTY_PRINT)) . "\n\n";
        file_put_contents($file, $content, FILE_APPEND);
    }

    public function sendOtp($phoneNumber, $otp = null)
    {
        $this->logToFile(['step' => 'sendOtp:start', 'phoneNumber' => $phoneNumber, 'otp_provided' => $otp !== null]);
        
        $otp = $otp ?? rand(100000, 999999); // Generate 6-digit OTP if not provided
        $brandName = 'Voiptella';//env('OTP_BRAND_NAME', 'CallChex');
        $validityMinutes = 15;//env('OTP_VALIDITY_MINUTES', 10);

        $this->logToFile(['step' => 'sendOtp:otp_generated', 'otp' => $otp, 'brandName' => $brandName]);

        // Check if number is Indian and route accordingly
        if ($this->isIndianNumber($phoneNumber)) {
            $brandName = 'LeadminePro';
            $this->logToFile(['step' => 'sendOtp:routing', 'provider' => 'Fast2SMS', 'phoneNumber' => $phoneNumber]);
            return $this->sendOtpViaFast2Sms($phoneNumber, $otp, $brandName);
        }

        $this->logToFile(['step' => 'sendOtp:routing', 'provider' => 'Plivo', 'phoneNumber' => $phoneNumber]);

        $client = new RestClient(env('PLIVO_AUTH_ID'),env('PLIVO_AUTH_TOKEN'));

        $message = "$brandName: Your verification code is $otp. It's valid for $validityMinutes minutes. Do not share this code.";

        $this->logToFile(['step' => 'sendOtp:plivo_request', 'message' => $message]);

        try {
            $response = $client->messages->create(env('PLIVO_SOURCE_NUMBER'),
                [$phoneNumber],
                $message
            );


            if ($response instanceof MessageCreateResponse) {
                $result = [
                    'success' => true,
                    'otp' => $otp,
                    'message' => 'OTP sent successfully.',
                    'plivo_response' => [
                        'message_uuid' => $response->messageUuid,
                        'api_id' => $response->apiId,
                        'message' => $response->message
                    ]
                ];
                
                $this->logToFile(['step' => 'sendOtp:plivo_success', 'result' => $result]);
                return $result;
            } elseif ($response instanceof MessageCreateErrorResponse) {
                $result = [
                    'success' => false,
                    'message' => 'Plivo error: ' . ($response->error ?? 'Unknown error'),
                    'plivo_response' => (array) $response
                ];
                
                $this->logToFile(['step' => 'sendOtp:plivo_error', 'result' => $result]);
                return $result;
            }

            $result = [
                'success' => false,
                'message' => 'Unexpected response from Plivo.',
                'plivo_response' => (array) $response
            ];
            
            $this->logToFile(['step' => 'sendOtp:plivo_unexpected', 'result' => $result]);
            return $result;

        } catch (\Exception $e) {
            $this->logToFile(['step' => 'sendOtp:exception', 'error' => $e->getMessage()]);
            
            return [
                'success' => false,
                'message' => 'Exception while sending OTP: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Check if the phone number is Indian
     * 
     * @param string $phoneNumber
     * @return bool
     */
    private function isIndianNumber($phoneNumber): bool
    {
        // Remove any whitespace
        $phoneNumber = preg_replace('/\s+/', '', $phoneNumber);
        
        // Remove +91 or 91 prefix if present
        $cleanedNumber = preg_replace('/^(\+91|91)/', '', $phoneNumber);
        
        // Check if it's a valid Indian mobile number (10 digits, starts with 6-9)
        return preg_match('/^[6-9]\d{9}$/', $cleanedNumber) === 1;
    }

    /**
     * Send OTP via Fast2SMS for Indian numbers
     * 
     * @param string $phoneNumber
     * @param string $otp
     * @param string $brandName
     * @return array
     */
    private function sendOtpViaFast2Sms($phoneNumber, $otp, $brandName): array
    {
        $this->logToFile(['step' => 'fast2sms:request', 'phoneNumber' => $phoneNumber, 'otp' => $otp]);
        
        $controller = new Fast2SmsController();
        $smsService = app(\App\Services\Fast2SmsService::class);
        
        $result = $controller->sendOtpInternalToIndianNumber(
            $phoneNumber,
            $otp,
            $brandName,
            $smsService
        );

        // Convert the response format to match Plivo response format
        if ($result['success']) {
            $response = [
                'success' => true,
                'otp' => $otp,
                'message' => 'OTP sent successfully via Fast2SMS.'
            ];
            
            $this->logToFile(['step' => 'fast2sms:success', 'result' => $response]);
            return $response;
        }

        $response = [
            'success' => false,
            'message' => $result['error'] ?? 'Failed to send OTP via Fast2SMS'
        ];
        
        $this->logToFile(['step' => 'fast2sms:failure', 'result' => $response]);
        return $response;
    }
}
