<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class AddRazorpayFieldsSafeToOrdersAndPaymentTransactions extends Migration
{
    public function up()
    {
        /**
         * ORDERS TABLE — SAFE ADDITIONS ONLY
         */
        Schema::table('orders', function (Blueprint $table) {

            // Add ONLY if not already present — safe mode
            if (!Schema::hasColumn('orders', 'payment_method')) {
                $table->string('payment_method', 50)
                      ->nullable()
                      ->after('status');
            }

            if (!Schema::hasColumn('orders', 'order_type')) {
                $table->string('order_type', 50)
                      ->nullable()
                      ->after('payment_method');
            }

            if (!Schema::hasColumn('orders', 'gateway_order_id')) {
                $table->string('gateway_order_id', 100)
                      ->nullable()
                      ->after('order_type');
            }

            if (!Schema::hasColumn('orders', 'gateway_subscription_id')) {
                $table->string('gateway_subscription_id', 100)
                      ->nullable()
                      ->after('gateway_order_id');
            }
        });

        /**
         * PAYMENT TRANSACTIONS TABLE — ADD initiated SAFELY
         */
        Schema::table('payment_transactions', function (Blueprint $table) {

            // Only extend ENUM — DO NOT remove existing values
            // Keep existing defaults intact
            if (Schema::hasColumn('payment_transactions', 'status')) {
                DB::statement("ALTER TABLE payment_transactions
                    MODIFY status ENUM('initiated','failed','success')
                    NOT NULL DEFAULT 'success'");
            }

            // Expand response to longText ONLY if varchar — safe upgrade
            $columnType = DB::selectOne(
                "SELECT DATA_TYPE FROM INFORMATION_SCHEMA.COLUMNS
                 WHERE table_name='payment_transactions' AND column_name='response'"
            )->DATA_TYPE ?? null;

            if ($columnType !== 'longtext') {
                $table->longText('response')->nullable()->change();
            }
        });
    }

    public function down()
    {
        Schema::table('orders', function (Blueprint $table) {
            // Remove only what we added
            if (Schema::hasColumn('orders', 'payment_method')) {
                $table->dropColumn('payment_method');
            }
            if (Schema::hasColumn('orders', 'order_type')) {
                $table->dropColumn('order_type');
            }
            if (Schema::hasColumn('orders', 'gateway_order_id')) {
                $table->dropColumn('gateway_order_id');
            }
            if (Schema::hasColumn('orders', 'gateway_subscription_id')) {
                $table->dropColumn('gateway_subscription_id');
            }
        });

        Schema::table('payment_transactions', function (Blueprint $table) {
            // Revert ONLY our enum change
            DB::statement("ALTER TABLE payment_transactions
                MODIFY status ENUM('failed','success')
                NOT NULL DEFAULT 'success'");

            // Revert longText → varchar(255)
            $table->string('response', 255)->nullable()->change();
        });
    }
}
