<?php

namespace App\Http\Controllers;

use App\Helper\Helper;
use App\Mail\ContactEnquiryMail;
use App\Models\ContactUs;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class ContactUsController extends Controller
{
    /**
     * Display a listing of contact entries with optional search and pagination.
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */

    public function index(Request $request)
    {
        // Start with query builder (not get())
        $query = ContactUs::query();

        // Search filter
        if ($request->has('search') && $request->search !== null) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Pagination parameters
        $currentPage = max(1, (int) $request->input('page', 1));
        $limit = $request->input('limit', 10);
        $validLimits = [25, 50, 100];
        $limit = in_array($limit, $validLimits) ? $limit : 10;
        $offset = ($currentPage - 1) * $limit;

        // Total count
        $totalRows = $query->count();

        // Fetch paginated results
        // Fetch paginated results with order
        $contact = $query->orderBy('created_at', 'desc')
            ->skip($offset)
            ->take($limit)
            ->get();

        return view('Contact_us.index', [
            'contact' => $contact,
            'record_count' => $totalRows,
            'page' => $currentPage,
            'show' => $limit,
            'url' => $request->url(),
        ]);
    }


    /**
     * Store a new contact entry and send an email.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        try {
            Log::info('Contact form submission received', $request->all());

            // Validate request data
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'message' => 'required|string',
                'g-recaptcha-response' => 'required|captcha',
            ], [
                'g-recaptcha-response.required' => 'Please complete the reCAPTCHA verification.',
                'g-recaptcha-response.captcha' => 'reCAPTCHA verification failed. Please try again.',
            ]);

            if ($validator->fails()) {
                return $request->ajax()
                    ? response()->json(['success' => false, 'errors' => $validator->errors()], 422)
                    : redirect()->back()->withErrors($validator)->withInput()->with('error', 'Validation failed. Please check your input.');
            }
            $cleanPhone = preg_replace('/\D/', '', $request->phone);
            $fullPhone = $request->country_code . $cleanPhone;
            $payload = [
                'name'         => $request->name,
                'email'        => $request->email,
                'message'      => $request->message,
                'phone'        => $fullPhone,
            ];

            $url = env('API_URL') . "lead/contact";

            try {
                $apiResponse = Helper::PostApi($url, $payload);

                if (!$apiResponse->success) {
                    return $request->ajax()
                        ? response()->json(['success' => false, 'message' => $apiResponse->message], 500)
                        : redirect()->back()->with('error', $apiResponse->message)->withInput();
                }
            } catch (\Throwable $ex) {
                return $request->ajax()
                    ? response()->json(['success' => false, 'message' => $ex->getMessage()], 500)
                    : redirect()->back()->with('error', $ex->getMessage())->withInput();
            }

            return redirect()->back()->with('success', 'Your message has been sent successfully.');
        } catch (\Exception $e) {
            Log::error('Failed to save contact: ' . $e->getMessage());
            return $request->ajax()
                ? response()->json(['success' => false, 'message' => 'Failed to save data: ' . $e->getMessage()], 500)
                : redirect()->back()->with('error', 'Failed to save your message. Please try again.')->withInput();
        }
    }

    /**
     * Delete a contact entry.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        try {
            $contact = ContactUs::findOrFail($id);
            $contact->delete();
            Log::info('Contact deleted successfully', ['id' => $id]);

            return redirect()->back()->with('success', 'Contact deleted successfully.');
        } catch (\Exception $e) {
            Log::error('Failed to delete contact: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete contact. Please try again.');
        }
    }
}
