<?php

namespace App\Http\Controllers;

use App\Helper\Helper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\MessageBag;

class FrontendRegisterController extends Controller
{
    public function index()
    {
        $url = env('API_URL') . "leadmine-pro-new-register";

        $response = Helper::GetApi($url);

        $leads = $response->success ? $response->data : [];
        return view('new-register.index', compact('leads'));
    }

    /**
     * Show registration page
     */
    public function showRegisterForm()
    {
        $path = resource_path('data/full-country-codes.json');
        $countries = json_decode(file_get_contents($path), true);
        return view('new-register.register', compact('countries'));
    }


    /**
     * Handle registration and send EMAIL OTP
     */
    public function store(Request $request)
    {
        $errors = new MessageBag();
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'company_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'password' => 'required|min:6|confirmed',
            'g-recaptcha-response' => app()->environment('production') ? 'required|captcha' : '',
        ], [
            'g-recaptcha-response.required' => 'Please complete the reCAPTCHA verification.',
            'g-recaptcha-response.captcha' => 'reCAPTCHA verification failed. Please try again.',
        ]);
        $cleanPhone = preg_replace('/\D/', '', $request->phone);
        $fullPhone = $request->country_code . $cleanPhone;
        $payload = [
            'name' => $request->name,
            'email' => $request->email,
            'company_name' => $request->company_name,
            'password' => $request->password,
            'phone' => $fullPhone,
            'password_confirmation' => $request->password_confirmation,
        ];
        $url = env('API_URL') . "lead/register";

        try {
            $response = Helper::PostApi($url, $payload);

            if ($response->success) {
                Session::put('register_email', $request->email);

                return redirect()->route('frontend.verify_otp', [
                    'step' => 'email',
                    'email' => $request->email
                ]);
            }

            foreach ($response->errors ?? [] as $key => $msg) {
                $errors->add($key, $msg);
            }
        } catch (\Throwable $ex) {
            $errors->add('error', $ex->getMessage());
        }

        return back()->withInput()->with('errors', $errors);
    }



    /**
     * Show OTP page (step: email or mobile)
     */
    public function verifyOtp(Request $request)
    {
        $step = $request->step ?? 'email';
        $email = $request->email ?? Session::get('register_email');

        if (!$email) {
            return redirect()->route('register');
        }

        return view('new-register.verify-otp', compact('step', 'email'));
    }



    /**
     * Verify EMAIL OTP → then MOBILE OTP → success
     */
    public function verifyBothOtps(Request $request)
    {
        // STEP 1: VERIFY EMAIL OTP
        if ($request->step === 'email') {

            $payload = [
                'email' => $request->email,
                'email_otp' => $request->email_otp,
            ];

            $url = env('API_URL') . "lead/verify-email";

            try {
                $response = Helper::PostApi($url, $payload);

                if ($response->success) {
                    // MUST RETURN JSON for JS to switch to mobile section
                    return response()->json([
                        'success' => true,
                        'phone' => $response->phone ?? null,   // your API must send phone
                    ]);
                }

                return response()->json([
                    'success' => false,
                    'message' => $response->message ?? 'Invalid email OTP',
                ], 422);
            } catch (\Throwable $ex) {
                return response()->json([
                    'success' => false,
                    'message' => $ex->getMessage(),
                ], 500);
            }
        }


        // STEP 2: VERIFY MOBILE OTP
        if ($request->step === 'mobile') {
            $payload = [
                'email' => $request->email,
                'mobile_otp' => $request->otp,
            ];

            $url = env('API_URL') . "lead/verify-mobile";

            try {
                $response = Helper::PostApi($url, $payload);

                if ($response->success) {
                    return response()->json([
                        'success' => true
                    ]);
                }

                return response()->json([
                    'success' => false,
                    'message' => $response->message ?? 'Invalid  OTP'
                ], 422);
            } catch (\Throwable $ex) {
                return response()->json([
                    'success' => false,
                    'message' => $ex->getMessage()
                ], 500);
            }
        }


        // DEFAULT FALLBACK
        return response()->json([
            'success' => false,
            'message' => 'Invalid request'
        ], 422);
    }



    /**
     * Resend OTP (email or mobile)
     */
    public function resendOtp(Request $request)
    {
        // EMAIL RESEND
        if ($request->type === 'email') {
            $payload = ['email' => $request->email];
            $url = env('API_URL') . "lead/resend-email-otp";
            $apiResponse = Helper::PostApi($url, $payload);
            return response()->json([
                'success' => $apiResponse->success ?? false,
                'message' => $apiResponse->message ?? null
            ]);
        }

        // MOBILE RESEND
        if ($request->type === 'mobile') {
            $payload = [
                'email' => $request->email,
            ];
            $url = env('API_URL') . "lead/resend-mobile-otp";
            $apiResponse = Helper::PostApi($url, $payload);
            return response()->json([
                'success' => $apiResponse->success ?? false,
                'message' => $apiResponse->message ?? null
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Invalid resend type'
        ], 422);
    }



    /**
     * Thank you page
     */
    public function thankYou()
    {
        return view('new-register.thank-you');
    }
}
