<?php

namespace App\Http\Controllers;

use App\Helper\Helper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;

class RazorpayController extends Controller
{
    /**
     * Create Razorpay order for recharge (frontend proxy)
     */
    public function createOrderRecharge(Request $r)
    {
        Log::info('FRONT_RZP_RECHARGE_STEP_1: Incoming request', $r->all());

        $r->validate(['amount' => 'required|numeric|min:1']);
        Log::info('FRONT_RZP_RECHARGE_STEP_2: Validation OK', ['amount' => $r->amount]);

        $url = rtrim(env('API_URL'), '/') . '/razorpay-create-order-recharge';
        $body = [
            'id' => Session::get('id'),
            'token' => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),
            'amount' => $r->input('amount'),
        ];

        Log::info('FRONT_RZP_RECHARGE_STEP_3: Calling Backend API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);
            Log::info('FRONT_RZP_RECHARGE_STEP_4: Backend Response', ['response' => $response]);

            if (! isset($response->success) || ! $response->success) {
                Log::warning('FRONT_RZP_RECHARGE_STEP_5: Backend FAILURE', ['response' => $response]);

                return response()->json([
                    'success' => false,
                    'message' => $response->message ?? 'Failed to create order',
                ], 400);
            }

            Log::info('FRONT_RZP_RECHARGE_STEP_6: Returning success response');

            return response()->json([
                'success' => true,
                'key' => env('RAZORPAY_KEY'),
                'order_id' => $response->order_id ?? null,
                'amount' => $r->input('amount'),
            ]);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_RECHARGE_EXCEPTION', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Server error creating order',
            ], 500);
        }
    }

    /**
     * Verify recharge order (frontend proxy)
     */
    public function verifyOrderRecharge(Request $r)
    {
        Log::info('FRONT_RZP_VERIFY_RECHARGE_STEP_1: Incoming verify request', $r->all());

        $r->validate([
            'razorpay_order_id' => 'required|string',
            'razorpay_payment_id' => 'required|string',
            'razorpay_signature' => 'required|string',
            'amount' => 'required|numeric',
        ]);

        Log::info('FRONT_RZP_VERIFY_RECHARGE_STEP_2: Validation OK');

        $url = rtrim(env('API_URL'), '/') . '/razorpay-verify-order-recharge';
        $body = [
            'id' => Session::get('id'),
            'token' => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),
            'razorpay_order_id' => $r->input('razorpay_order_id'),
            'razorpay_payment_id' => $r->input('razorpay_payment_id'),
            'razorpay_signature' => $r->input('razorpay_signature'),
            'amount' => $r->input('amount'),
        ];

        Log::info('FRONT_RZP_VERIFY_RECHARGE_STEP_3: Calling Backend Verify API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);

            Log::info('FRONT_RZP_VERIFY_RECHARGE_STEP_4: Backend Verify Response', ['response' => $response]);

            return response()->json($response);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_VERIFY_RECHARGE_EXCEPTION', [
                'error' => $e->getMessage(),
                'body' => $body,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Server error verifying payment',
            ], 500);
        }
    }

    /**
     * Create subscription (frontend proxy)
     */
    public function createSubscription(Request $r)
    {
        Log::info('FRONT_RZP_SUB_STEP_1: Incoming subscription create request', $r->all());

        $r->validate(['plan_id' => 'required|string']);
        Log::info('FRONT_RZP_SUB_STEP_2: Validation OK');

        $url = rtrim(env('API_URL'), '/') . '/razorpay-create-subscription';
        $body = [
            'id' => Session::get('id'),
            'token' => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),
            'plan_id' => $r->input('plan_id'),
            'total_count' => $r->input('total_count', 12),
        ];

        Log::info('FRONT_RZP_SUB_STEP_3: Calling Backend API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);
            Log::info('FRONT_RZP_SUB_STEP_4: Backend Response', ['response' => $response]);

            if (! isset($response->success) || ! $response->success) {
                Log::warning('FRONT_RZP_SUB_STEP_5: Backend FAILURE', ['response' => $response]);

                return response()->json([
                    'success' => false,
                    'message' => $response->message ?? 'Failed to create subscription',
                ], 400);
            }

            Log::info('FRONT_RZP_SUB_STEP_6: Returning subscription success response');

            return response()->json([
                'success' => true,
                'key' => env('RAZORPAY_KEY'),
                'subscription_id' => $response->subscription_id ?? null,
                'order_db_id' => $response->order_db_id ?? null,
            ]);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_SUB_EXCEPTION', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Server error creating subscription',
            ], 500);
        }
    }

    /**
     * Verify subscription (frontend proxy)
     */
    public function verifySubscription(Request $r)
    {
        Log::info('FRONT_RZP_VERIFY_SUB_STEP_1: Incoming subscription verify request', $r->all());

        $r->validate([
            'razorpay_subscription_id' => 'required|string',
            'razorpay_payment_id' => 'required|string',
            'razorpay_signature' => 'required|string',
        ]);

        Log::info('FRONT_RZP_VERIFY_SUB_STEP_2: Validation OK');

        $url = rtrim(env('API_URL'), '/') . '/razorpay-verify-subscription';
        $body = [
            'id' => Session::get('id'),
            'token' => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),
            'razorpay_subscription_id' => $r->input('razorpay_subscription_id'),
            'razorpay_payment_id' => $r->input('razorpay_payment_id'),
            'razorpay_signature' => $r->input('razorpay_signature'),
        ];

        Log::info('FRONT_RZP_VERIFY_SUB_STEP_3: Calling Backend Verify API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);

            Log::info('FRONT_RZP_VERIFY_SUB_STEP_4: Backend Verify Response', ['response' => $response]);

            return response()->json($response);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_VERIFY_SUB_EXCEPTION', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Server error verifying subscription',
            ], 500);
        }
    }

    /**
     * Create Auto-Recharge Mandate (frontend proxy for UPI setup)
     */
    public function createAutoRechargeMandate(Request $r)
    {
        Log::info('FRONT_RZP_AUTO_MANDATE_STEP_1: Incoming mandate request', $r->all());

        $r->validate(['amount' => 'required|numeric|min:1']);
        Log::info('FRONT_RZP_AUTO_MANDATE_STEP_2: Validation OK', ['amount' => $r->amount]);

        $url = rtrim(env('API_URL'), '/') . '/razorpay-create-auto-recharge-mandate';

        $body = [
            'id'       => Session::get('id'),
            'token'    => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),
            'amount'   => $r->input('amount'),
        ];

        Log::info('FRONT_RZP_AUTO_MANDATE_STEP_3: Calling Backend API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);
            Log::info('FRONT_RZP_AUTO_MANDATE_STEP_4: Backend Response', ['response' => $response]);

            if (!isset($response->success) || !$response->success) {
                Log::warning('FRONT_RZP_AUTO_MANDATE_STEP_5: Backend FAILURE', ['response' => $response]);

                return response()->json([
                    'success' => false,
                    'message' => $response->message ?? 'Failed to create mandate',
                ], 400);
            }

            // Backend now returns: order_id + customer_id + key
            return response()->json([
                'success'      => true,
                'key'          => $response->razorpay_key,
                'order_id'     => $response->order_id,
                'customer_id'  => $response->customer_id,
                'amount'       => $r->input('amount'),
            ]);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_AUTO_MANDATE_EXCEPTION', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Server error creating mandate',
            ], 500);
        }
    }



    /**
     * Verify/Save Auto-Recharge Mandate (frontend proxy post-handler)
     */
    public function verifyAutoRechargeMandate(Request $r)
    {
        Log::info('FRONT_RZP_VERIFY_AUTO_MANDATE_STEP_1: Incoming verify request', $r->all());

        $r->validate([
            'razorpay_payment_id' => 'required|string',
            'razorpay_signature'  => 'required|string',
            'razorpay_order_id'   => 'required|string',
            'auto_reload_threshold' => 'required|numeric|min:1',
            'auto_reload_amount'    => 'required|numeric|min:1',
        ]);

        Log::info('FRONT_RZP_VERIFY_AUTO_MANDATE_STEP_2: Validation OK');

        $url = rtrim(env('API_URL'), '/') . '/razorpay-verify-auto-recharge-mandate';

        $body = [
            'id'       => Session::get('id'),
            'token'    => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),

            'razorpay_payment_id' => $r->razorpay_payment_id,
            'razorpay_signature'  => $r->razorpay_signature,
            'razorpay_order_id'   => $r->razorpay_order_id,

            'auto_reload_threshold' => $r->auto_reload_threshold,
            'auto_reload_amount'    => $r->auto_reload_amount,
        ];

        Log::info('FRONT_RZP_VERIFY_AUTO_MANDATE_STEP_3: Calling Backend Verify API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);

            Log::info('FRONT_RZP_VERIFY_AUTO_MANDATE_STEP_4: Backend Response', ['response' => $response]);

            return response()->json($response);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_VERIFY_AUTO_MANDATE_EXCEPTION', [
                'error' => $e->getMessage(),
                'body' => $body,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Server error saving mandate',
            ], 500);
        }
    }


    /**
     * Trigger Auto-Recharge (frontend proxy for low-balance trigger)
     */
    public function triggerAutoRecharge(Request $r)
    {
        Log::info('FRONT_RZP_AUTO_TRIGGER_STEP_1: Incoming trigger request', $r->all());

        $r->validate(['amount' => 'required|numeric|min:1']);
        Log::info('FRONT_RZP_AUTO_TRIGGER_STEP_2: Validation OK', ['amount' => $r->amount]);

        $url = rtrim(env('API_URL'), '/') . '/razorpay-trigger-auto-recharge';

        $body = [
            'id'       => Session::get('id'),
            'token'    => Session::get('tokenId'),
            'parentId' => Session::get('parentId'),
            'amount'   => $r->input('amount'),
        ];

        Log::info('FRONT_RZP_AUTO_TRIGGER_STEP_3: Calling Backend API', ['url' => $url, 'body' => $body]);

        try {
            $response = Helper::PostApi($url, $body);

            Log::info('FRONT_RZP_AUTO_TRIGGER_STEP_4: Backend Response', ['response' => $response]);

            if (!isset($response->success) || !$response->success) {
                Log::warning('FRONT_RZP_AUTO_TRIGGER_STEP_5: Backend FAILURE', ['response' => $response]);

                return response()->json([
                    'success' => false,
                    'message' => $response->message ?? 'Failed to trigger auto recharge',
                ], 400);
            }

            // Backend returns payment_id
            return response()->json([
                'success'    => true,
                'payment_id' => $response->payment_id ?? null,
                'status'     => 'processing',
            ]);
        } catch (\Throwable $e) {
            Log::error('FRONT_RZP_AUTO_TRIGGER_EXCEPTION', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Server error triggering recharge',
            ], 500);
        }
    }
}
