<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Razorpay\Api\Api;
use Razorpay\Api\Entity;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Request as FacadesRequest;

class TestController extends Controller
{
    protected function api()
    {
        return new Api(env('RAZORPAY_KEY'), env('RAZORPAY_SECRET'));
    }

    /**
     * Create UPI Autopay mandate (test) - KEY FIX: No customer creation/param (auto-handled by Razorpay).
     */
    public function createUpiAutopay(Request $r)
    {
        try {
            $api = $this->api();
            $amount = intval($r->amount ?? 1000);
            if ($amount <= 0) {
                return response()->json([
                    'error' => true,
                    'message' => 'Invalid amount'
                ], 422);
            }

            // Create temporary plan for setup
            $plan = $api->plan->create([
                'period' => 'monthly',
                'interval' => 1,
                'item' => [
                    'name' => 'Auto Recharge Setup',
                    'amount' => $amount * 100, // paise
                    'currency' => 'INR'
                ]
            ]);

            // Create setup subscription (for mandate auth) - FIX: No 'customer' param
            $sub = $api->subscription->create([
                'plan_id' => $plan->id,
                'total_count' => 12,
                'customer_notify' => 1,
                'notes' => ['type' => 'auto_recharge_setup']
            ]);

            Log::info("Setup Sub Created: Sub " . $sub->id . " | Plan: " . $plan->id . " | Customer auto-created on auth");

            return response()->json([
                'error' => false,
                'key' => env('RAZORPAY_KEY'),
                'subscription_id' => $sub->id,
                'amount' => $amount
            ], 200);

        } catch (\Exception $e) {
            Log::error("createUpiAutopay error: " . $e->getMessage());
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Trigger auto-recharge - FIX: No 'customer' param; creates independent one-off sub (re-auth required for full test).
     */
    public function triggerAutoRecharge(Request $r)
    {
        try {
            $simulate = FacadesRequest::get('simulate', false) || $r->simulate;
            $api = $this->api();
            $rechargeAmount = intval($r->amount ?? 20000);

            if ($simulate) {
                Log::info("SIMULATED Recharge");
                return response()->json([
                    'success' => true,
                    'message' => 'Charge simulated (no real debit)',
                    'subscription_id' => 'sim_' . uniqid(),
                    'status' => 'success'
                ], 200);
            }

            // Create one-off plan
            $plan = $api->plan->create([
                'period' => 'monthly',
                'interval' => 1,
                'item' => [
                    'name' => 'Auto Recharge Top-up',
                    'amount' => $rechargeAmount,
                    'currency' => 'INR'
                ]
            ]);

            // Create one-off subscription (triggers charge; customer auto-created if new)
            $sub = $api->subscription->create([
                'plan_id' => $plan->id,
                'total_count' => 1,
                'customer_notify' => 0,
                'notes' => ['type' => 'auto_recharge']
            ]);

            Log::info("One-Off Recharge Sub Created: Sub " . $sub->id . " | Expected Charge: ₹" . ($rechargeAmount / 100));

            return response()->json([
                'success' => true,
                'message' => 'Recharge subscription created—UPI debit initiated (re-auth may be needed)',
                'subscription_id' => $sub->id,
                'status' => $sub->status
            ], 200);

        } catch (\Exception $e) {
            Log::error("triggerAutoRecharge error: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
