<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Helper\Helper;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;

class BalanceCheck
{
    public function handle(Request $request, Closure $next)
    {
        $currentDate = new \DateTime();

        try {

            /* -------------------------------------------------------
             * STEP 1 — Wallet Balance
             * ------------------------------------------------------- */
            $urlWalletBalance = env('API_URL') . 'wallet/balance';
            $responseWalletBalance = Helper::GetApi($urlWalletBalance);

            $prevWalletBalance = floor($responseWalletBalance?->data?->amount ?? 0);
            $walletBalance = $prevWalletBalance;


            /* -------------------------------------------------------
             * STEP 1.1 — Balance value change detection
             * ------------------------------------------------------- */
            $lastShownBalance = Session::get('lastShownWalletBalance');
            $balanceValueChanged = ($lastShownBalance === null || $lastShownBalance != $walletBalance);
            Session::put('lastShownWalletBalance', $walletBalance);


            /* -------------------------------------------------------
             * STEP 1.2 — Recharge tracking
             * ------------------------------------------------------- */
            $lastRechargeTime = Session::get('lastAutoRechargeTime');
            $lastRechargeBalance = Session::get('lastAutoRechargeBalance');

            $balanceChangedSinceRecharge = (
                $lastRechargeBalance === null ||
                $lastRechargeBalance != $walletBalance
            );


            /* -------------------------------------------------------
             * STEP 2 — Active Plans
             * ------------------------------------------------------- */
            $urlClientPackages = env('API_URL') . 'active-client-plans';
            $responseClientPackages = Helper::GetApi($urlClientPackages);
            $packageMessage = '';

            if ($responseClientPackages->success ?? false) {
                $clientPlans = $responseClientPackages?->data?->{Session::get('parentId')} ?? [];
                foreach ($clientPlans as $package) {
                    $endDate = new \DateTime($package->end_time);
                    $interval = $currentDate->diff($endDate);
                    $daysLeft = $interval->days;

                    if ($interval->invert === 0 && $daysLeft <= 3) {
                        $packageMessage = "Your {$package->package_name} plan will expire in {$daysLeft} day(s).";
                        break;
                    }
                }
            }


            /* -------------------------------------------------------
             * STEP 3 — User Details
             * ------------------------------------------------------- */
            $url = env('API_URL') . 'user-detail';
            $body = [
                'id' => Session::get('id'),
                'token' => Session::get('tokenId'),
                'parentId' => Session::get('parentId')
            ];
            $responseUserDetail = Helper::PostApi($url, $body);
            $user = $responseUserDetail->data ?? null;

            $autoReloadEnabled = $user->auto_reload_enabled ?? 0;
            $autoReloadThreshold = $user->auto_reload_threshold ?? null;
            $autoReloadAmount = $user->auto_reload_amount ?? null;

            $autoReloadMessage = '';
            $autoReloadSucceeded = false;


            /* -------------------------------------------------------
             * STEP 4 — Auto Reload with anti-repeat
             * ------------------------------------------------------- */
            $lowBalance = ($autoReloadThreshold !== null && $walletBalance <= $autoReloadThreshold);

            if ($autoReloadEnabled && $autoReloadThreshold && $lowBalance) {

                // 1. Prevent repeated recharge due to refresh
                if ($lastRechargeTime && now()->diffInMinutes($lastRechargeTime) < 3) {
                    $autoReloadMessage = "Auto-reload pending. Please wait.";
                }
                // 2. Prevent retry if balance didn't change since last recharge
                elseif (!$balanceChangedSinceRecharge) {
                    $autoReloadMessage = "Auto-reload recently applied.";
                } else {
                    try {
                        // Razorpay AUTO-RELOAD PROCESS--------------------------------------------------------
                        if (session::get('currency') == "₹") {
                            $paymentBody = [
                                'id'    => Session::get('id'),
                                'token' => Session::get('tokenId'),
                                'parentId' => Session::get('parentId'),
                                'amount' => $autoReloadAmount, // IMPORTANT: backend expects INR, not paise
                            ];

                            $urlTrigger = env('API_URL') . 'razorpay-trigger-auto-recharge';
                            $responsePayment = Helper::PostApi($urlTrigger, $paymentBody);

                            if (!empty($responsePayment->success) && $responsePayment->success === true) {

                                // Backend sends: payment_id
                                $paymentId = $responsePayment->payment_id ?? null;

                                $autoReloadMessage = "Auto-reload successful: ₹{$autoReloadAmount} has been added to your wallet.";
                                $walletBalance += $autoReloadAmount;
                                $autoReloadSucceeded = true;

                                Session::put('lastAutoRechargeTime', now());
                                Session::put('lastAutoRechargeBalance', $walletBalance);
                            } else {

                                $error = $responsePayment->message ?? "Unknown error";

                                $autoReloadMessage = "Auto-reload failed: {$error}. Please recharge manually.";
                            }
                        } else {
                            // ENDS-----------------------------------------------------------------------------------
                            // STRIPE AUTO-RELOAD PROCESS-----------------------------------------------------------
                            // get default payment method
                            $paymentBody = [
                                'id' => Session::get('id'),
                                'token' => Session::get('tokenId'),
                            ];
                            $urlGetPaymentMethod = env('API_URL') . 'stripe/get-default-payment-method';
                            $responsePayment = Helper::PostApi($urlGetPaymentMethod, $paymentBody);
                            $defaultPayment = $responsePayment->data ?? null;

                            if ($defaultPayment) {
                                // trigger checkout
                                $checkoutBody = [
                                    'id' => Session::get('id'),
                                    'token' => Session::get('tokenId'),
                                    'full_name' => $user->name ?? 'Auto Reload User',
                                    'line1' => $user->address_line1 ?? 'N/A',
                                    'city' => $user->city ?? 'N/A',
                                    'state' => $user->state ?? 'N/A',
                                    'country' => $user->country ?? 'US',
                                    'postal_code' => $user->postal_code ?? '00000',
                                    'amount' => $autoReloadAmount,
                                    'payment_method' => $defaultPayment,
                                    'request_type' => 'recharge'
                                ];

                                $urlCheckout = env('API_URL') . 'checkout';
                                $responseCheckout = Helper::PostApi($urlCheckout, $checkoutBody);

                                if ($responseCheckout->success ?? false) {

                                    $autoReloadMessage = "Auto-reload successful: \${$autoReloadAmount} has been added to your wallet.";
                                    $walletBalance += $autoReloadAmount;  // update local state
                                    $autoReloadSucceeded = true;

                                    // record recharge meta
                                    Session::put('lastAutoRechargeTime', now());
                                    Session::put('lastAutoRechargeBalance', $walletBalance);
                                } else {
                                    $autoReloadMessage = "Auto-reload attempted but failed. Please recharge manually.";
                                }
                            } else {
                                $autoReloadMessage = "Auto-reload failed: No default payment method found.";
                            }
                        }
                        // ------------------------------------------------------------------------------------

                    } catch (\Exception $e) {
                        Log::error('Auto-reload failed: ' . $e->getMessage());
                        $autoReloadMessage = "Auto-reload error occurred. Please recharge manually.";
                    }
                }
            } else if ($lowBalance && !$autoReloadEnabled) {
                $autoReloadMessage = "Auto-reload is not enabled. Please recharge manually.";
            }


            /* -------------------------------------------------------
             * STEP 5 — If balance rises above threshold, reset lock
             * ------------------------------------------------------- */
            if ($balanceChangedSinceRecharge && $autoReloadThreshold !== null && $walletBalance > $autoReloadThreshold) {
                Session::forget('lastAutoRechargeBalance');
                Session::forget('lastAutoRechargeTime');
            }


            /* -------------------------------------------------------
             * STEP 6 — Build modal message (with old & new balance)
             * ------------------------------------------------------- */
            if ($autoReloadSucceeded) {
                $balanceMessage = "Your wallet balance was low (\$" . number_format($prevWalletBalance, 2) . "). "
                    . "Current balance: \$" . number_format($walletBalance, 2) . ".";
            } elseif ($lowBalance) {
                $balanceMessage = "Your wallet balance is low (\$" . number_format($walletBalance, 2) . ").";
            } else {
                $balanceMessage = null;
            }

            // Modal title logic
            if ($autoReloadSucceeded) {
                $modalTitle = 'Auto-reload Successful';
            } elseif ($lowBalance) {
                $modalTitle = 'Low Wallet Balance';
            } elseif (!empty($packageMessage)) {
                $modalTitle = 'Plan Expiry Notice';
            } else {
                $modalTitle = 'Account Notice';
            }

            $messages = array_filter([
                $balanceMessage,
                $packageMessage,
                $autoReloadMessage,
            ]);


            /* -------------------------------------------------------
             * STEP 7 — Modal throttle (5 minutes)
             * ------------------------------------------------------- */
            if (!empty($messages)) {

                $lastRun = Session::get('balanceCheckLastRun');
                $lastLow = Session::get('balanceCheckLastLowState');
                $currentLow = $lowBalance;

                Session::put('balanceCheckLastLowState', $currentLow);

                $stateChanged = ($lastLow === null || $lastLow !== $currentLow);

                // FIXED throttle: ALSO requires balance not changed
                if (
                    $lastRun &&
                    now()->diffInMinutes($lastRun) < 5 &&
                    $currentLow &&
                    !$stateChanged &&
                    !$balanceValueChanged
                ) {
                    // suppress modal only
                } else {
                    // show modal
                    $request->session()->flash('showBalanceModal', true);
                    $request->session()->flash('modalTitle', $modalTitle);
                    $request->session()->flash('modalMessage', implode(' ', $messages));

                    Session::put('balanceCheckLastRun', now());
                }
            } else {
                Session::forget('balanceCheckLastRun');
            }
        } catch (\Exception $e) {
            Log::error('BalanceCheck error: ' . $e->getMessage());
        }

        return $next($request);
    }
}
