<!DOCTYPE html>
<html>
<head>
    <title>Auto-Recharge Testing Workflow</title>
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        body { padding: 40px; font-family: sans-serif; max-width: 600px; }
        .section { margin-bottom: 24px; padding: 16px; border: 1px solid #ddd; border-radius: 8px; }
        input, button { padding: 10px; margin: 5px; }
        button { background: #4285f4; color: white; border: none; border-radius: 4px; cursor: pointer; }
        button:disabled { background: #ccc; cursor: not-allowed; }
        .hidden { display: none; }
        label { display: block; margin-bottom: 8px; font-weight: bold; }
        small { color: #666; }
    </style>
</head>

<body>
    <h2>Auto-Recharge Testing Workflow</h2>
    <small>Open browser console for logs/responses. Use test mode in Razorpay Dashboard.</small>

    {{-- Consent Section --}}
    <div id="consent-section" class="section">
        <h3>Step 1: Get User Consent</h3>
        <button id="enable-consent" class="btn-primary">Enable Auto-Recharge</button>
        <div id="consent-form" class="hidden">
            <label>
                <input type="checkbox" id="consent-checkbox" required>
                I consent to auto-recharge: When balance < ₹100, debit ₹200 via UPI Autopay mandate (RBI compliant, revocable anytime).
            </label>
            <div style="margin: 10px 0; padding: 10px; background: #f9f9f9; border-radius: 4px;">
                <small>Terms: One-time setup. Charges only on low balance. You can disable in settings.</small>
            </div>
            <button id="submit-consent">Set Up UPI Autopay Mandate</button>
            <input type="number" id="consent-amount" placeholder="Recharge Amount (e.g., 200)" value="200" style="width: 200px;">
        </div>
    </div>

    {{-- Balance Simulation & Trigger Section --}}
    <div id="trigger-section" class="section hidden">
        <h3>Step 2: Simulate Low Balance & Trigger</h3>
        <label>Simulated Current Balance (₹): <input type="number" id="sim-balance" value="150" min="0"></label>
        <label>Threshold (₹): <input type="number" id="threshold" value="100" readonly></label>
        <label>Recharge Amount (₹): <input type="number" id="recharge-amount" value="200" readonly></label>
        <p id="balance-status" style="color: green;">Balance OK</p>
        <button id="trigger-recharge" disabled>Trigger Auto-Recharge (Balance Low)</button>
        <div id="trigger-result"></div>
    </div>

    <script>
        const csrf = document.querySelector('meta[name="csrf-token"]').content;
        let subscriptionId = null; // Setup sub for reference

        // Consent Flow
        document.getElementById("enable-consent").onclick = () => {
            document.getElementById("consent-form").classList.remove("hidden");
        };

        document.getElementById("submit-consent").onclick = async () => {
            if (!document.getElementById("consent-checkbox").checked) {
                alert("Please check the consent box.");
                return;
            }

            const amount = parseInt(document.getElementById("consent-amount").value) || 200;

            try {
                const res = await fetch("/test/create-upi-autopay", {
                    method: "POST",
                    headers: { "Content-Type": "application/json", "X-CSRF-TOKEN": csrf },
                    body: JSON.stringify({ amount })
                });

                const text = await res.text();
                console.log("Consent Setup Response:", text);

                let data;
                try { data = JSON.parse(text); } catch (err) {
                    alert("Server error—check console.");
                    return;
                }

                if (data.error) {
                    alert("Setup failed: " + data.message);
                    return;
                }

                // Save setup sub ID
                subscriptionId = data.subscription_id;
                sessionStorage.setItem('consentGranted', 'true');
                sessionStorage.setItem('subId', subscriptionId);

                // Open Razorpay for Mandate (UPI only)
                const rzp = new Razorpay({
                    key: data.key,
                    subscription_id: data.subscription_id,
                    recurring: true,
                    method: "upi",
                    name: "Auto Recharge Consent",
                    description: "UPI Autopay Setup (Test)",
                    handler: function (resp) {
                        console.log("Mandate Created:", resp);
                        alert("Consent Granted! Mandate Active. Proceed to Step 2.");
                        document.getElementById("consent-section").innerHTML += '<p style="color: green;">✓ Consent & Mandate Set Up (Sub: ' + subscriptionId + ')</p>';
                        document.getElementById("trigger-section").classList.remove("hidden");
                        checkBalance(); // Initial check
                    },
                    modal: { ondismiss: () => console.log("Mandate dismissed") }
                });

                rzp.open();

            } catch (err) {
                console.error("Consent error:", err);
                alert("Network error.");
            }
        };

        // Balance Sim & Trigger
        function checkBalance() {
            const balance = parseInt(document.getElementById("sim-balance").value) || 0;
            const threshold = parseInt(document.getElementById("threshold").value) || 100;
            const statusEl = document.getElementById("balance-status");
            const triggerBtn = document.getElementById("trigger-recharge");

            if (balance < threshold) {
                statusEl.textContent = "⚠️ Low Balance Detected!";
                statusEl.style.color = "red";
                triggerBtn.disabled = false;
            } else {
                statusEl.textContent = "Balance OK";
                statusEl.style.color = "green";
                triggerBtn.disabled = true;
            }
        }

        document.getElementById("sim-balance").oninput = checkBalance;

        document.getElementById("trigger-recharge").onclick = async () => {
            if (!subscriptionId) {
                alert("No setup—complete consent first.");
                return;
            }

            const amount = (parseInt(document.getElementById("recharge-amount").value) || 200) * 100; // Paise
            const resultEl = document.getElementById("trigger-result");

            try {
                const res = await fetch("/test/trigger-auto-recharge", {
                    method: "POST",
                    headers: { "Content-Type": "application/json", "X-CSRF-TOKEN": csrf },
                    body: JSON.stringify({
                        amount
                    })
                });

                const data = await res.json();
                console.log("Trigger Response:", data);

                if (data.success) {
                    resultEl.innerHTML = `<p style="color: green;">✓ Auto-Recharge Triggered! Balance +₹200 (Simulated). Sub ID: ${data.subscription_id}</p>`;
                    const newBalance = parseInt(document.getElementById("sim-balance").value) + 200;
                    document.getElementById("sim-balance").value = newBalance;
                    checkBalance();
                    alert("Success! In real app, webhook would update wallet. Check Dashboard for new sub.");
                } else {
                    resultEl.innerHTML = `<p style="color: red;">✗ Failed: ${data.message}</p>`;
                    alert("Trigger failed—check Razorpay logs.");
                }

            } catch (err) {
                console.error("Trigger error:", err);
                resultEl.innerHTML = `<p style="color: red;">Network error.</p>`;
            }
        };

        // On load, check session for prior consent
        if (sessionStorage.getItem('consentGranted') === 'true') {
            subscriptionId = sessionStorage.getItem('subId');
            document.getElementById("consent-section").innerHTML += '<p style="color: green;">Consent Already Granted—Skip to Step 2. (Sub: ' + subscriptionId + ')</p>';
            document.getElementById("trigger-section").classList.remove("hidden");
            checkBalance();
        }
    </script>
</body>
</html>
